import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.Random;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.LineBorder;

public class MineWalkerPanel extends JPanel {
    private static final int DEFAULT_MINE_PERCENTAGE = 30;
    private static final int STARTING_LIVES = 5;
    private static final int STARTING_SCANS = 3;

    private static int GRID_SIZE;

    private static TileButton[][] board;
    private static ArrayList<Point> path;
    private static TileButton currentTile;
    private static int score;
    private static int moves;
    private static int scans;
    private static int lives;

    private JLabel livesLabel;
    private JTextField minePercentageField;
    JButton sonarScanButton;

    public MineWalkerPanel(int width, int height) {
        setLayout(new BorderLayout(height, width));

        setFocusable(true);
        requestFocusInWindow();

        GRID_SIZE = 10;

        board = new TileButton[GRID_SIZE][GRID_SIZE];
        path = RandomPath.getPath(GRID_SIZE);
        score = 0;
        moves = 0;
        scans = STARTING_SCANS;
        lives = STARTING_LIVES;

        JPanel gameGridPanel = createGameGridPanel();
        JPanel livesPanel = createLivesPanel();
        JPanel statusPanel = createStatusPanel();
        JPanel controlsPanel = createControlsPanel();

        add(gameGridPanel, BorderLayout.CENTER);
        add(livesPanel, BorderLayout.NORTH);
        add(statusPanel, BorderLayout.SOUTH);
        add(controlsPanel, BorderLayout.WEST);
        
        initializeGame();

        this.addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if (!minePercentageField.hasFocus()) {
                    move(e);
                }
            }
        });
    }

    private void move(KeyEvent e) {
        TileButton clickedTile = board[currentTile.getLocation().x][currentTile.getLocation().y];
        switch (e.getKeyCode()) {
            case 87 -> clickedTile = board[currentTile.getLocation().y - 1][currentTile.getLocation().x]; // W key
            case 38 -> clickedTile = board[currentTile.getLocation().y - 1][currentTile.getLocation().x]; // Up arrow
            case 65 -> clickedTile = board[currentTile.getLocation().y][currentTile.getLocation().x - 1]; // A key
            case 37 -> clickedTile = board[currentTile.getLocation().y][currentTile.getLocation().x - 1]; // Left arrow
            case 83 -> clickedTile = board[currentTile.getLocation().y + 1][currentTile.getLocation().x]; // S key
            case 40 -> clickedTile = board[currentTile.getLocation().y + 1][currentTile.getLocation().x]; // Down arrow
            case 68 -> clickedTile = board[currentTile.getLocation().y][currentTile.getLocation().x + 1]; // D key
            case 39 -> clickedTile = board[currentTile.getLocation().y][currentTile.getLocation().x + 1]; // Right arrow
            default -> { return; }
        }

        if (clickedTile.isMine()) {
            clickedTile.setHidden(false);
            clickedTile.setEnabled(false);
            clickedTile.setBackground(Color.BLACK);
            lives--;
            if (lives <= 0) {
                JOptionPane.showMessageDialog(MineWalkerPanel.this, "Game Over! Final score: " + getScore());
                revealAllMines();
                toggleAllTiles(false);
            }
        } else {
            exposeTile(clickedTile);
            currentTile.setText("");
            currentTile = clickedTile;
            clickedTile.setText("X");
            moves++;
            if (clickedTile.getLocation().x == GRID_SIZE-1 && clickedTile.getLocation().y == GRID_SIZE-1) {
                JOptionPane.showMessageDialog(MineWalkerPanel.this, "You Win! Final score: " + getScore());
                revealAllMines();
                toggleAllTiles(false);
            }
        }

        livesLabel.setText("Lives: " + lives);
    }

    private JPanel createGameGridPanel() {
        JPanel gridPanel = new JPanel();
        gridPanel.setLayout(new java.awt.GridLayout(GRID_SIZE, GRID_SIZE));

        for (int row = 0; row < GRID_SIZE; row++) {
            for (int col = 0; col < GRID_SIZE; col++) {
                TileButton tile = new TileButton(row, col);
                tile.setBackground(Color.GRAY);
                tile.addActionListener(new TileButtonListener());
                board[row][col] = tile;
                gridPanel.add(tile);
            }
        }

        return gridPanel;
    }

    private JPanel createLivesPanel() {
        JPanel livesPanel = new JPanel();
        livesPanel.setLayout(new FlowLayout());

        minePercentageField = new JTextField(5);
        minePercentageField.setText(String.valueOf(DEFAULT_MINE_PERCENTAGE));
        minePercentageField.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent e) {
                char c = e.getKeyChar();
                if (!((c >= '0') && (c <= '9') ||
                    (c == KeyEvent.VK_BACK_SPACE) ||
                    (c == KeyEvent.VK_DELETE))) {
                    getToolkit().beep();
                    e.consume();
                }
            }
        });
        livesPanel.add(minePercentageField);

        JButton newGameButton = new JButton("New Game");
        newGameButton.addActionListener(new NewGameListener());
        livesPanel.add(newGameButton);

        return livesPanel;
    }

    private JPanel createStatusPanel() {
        JPanel statusPanel = new JPanel();
        statusPanel.setLayout(new FlowLayout());

        livesLabel = new JLabel("Lives: " + lives);
        statusPanel.add(livesLabel);

        return statusPanel;
    }

    private JPanel createControlsPanel() {
        JPanel controlsPanel = new JPanel();
        controlsPanel.setLayout(new BoxLayout(controlsPanel, BoxLayout.Y_AXIS));

        JLabel colorCodeLabel = new JLabel("0 Mines Adjacent");
        colorCodeLabel.setBackground(Color.GREEN);
        colorCodeLabel.setOpaque(true);
        controlsPanel.add(colorCodeLabel);

        colorCodeLabel = new JLabel("1 Mine Adjacent");
        colorCodeLabel.setBackground(Color.YELLOW);
        colorCodeLabel.setOpaque(true);
        controlsPanel.add(colorCodeLabel);
        
        colorCodeLabel = new JLabel("2 Mines Adjacent");
        colorCodeLabel.setBackground(Color.ORANGE);
        colorCodeLabel.setOpaque(true);
        controlsPanel.add(colorCodeLabel);

        colorCodeLabel = new JLabel("3 Mines Adjacent");
        colorCodeLabel.setBackground(Color.RED);
        colorCodeLabel.setOpaque(true);
        controlsPanel.add(colorCodeLabel);

        colorCodeLabel = new JLabel("Mine Activated");
        colorCodeLabel.setBackground(Color.BLACK);
        colorCodeLabel.setForeground(Color.WHITE);
        colorCodeLabel.setOpaque(true);
        controlsPanel.add(colorCodeLabel);

        colorCodeLabel = new JLabel("X <- Your Player");
        colorCodeLabel.setBorder(new LineBorder(Color.BLACK, 2));
        colorCodeLabel.setOpaque(true);
        controlsPanel.add(colorCodeLabel);

        sonarScanButton = new JButton("Remaining scans: " + scans);
        sonarScanButton.addActionListener(new SonarButtonListener());
        controlsPanel.add(sonarScanButton);

        return controlsPanel;
    }

    private void initializeGame() {
        toggleAllTiles(true);
        
        for (int row = 0; row < GRID_SIZE; row++) {
            for (int col = 0; col < GRID_SIZE; col++) {
                TileButton tile = board[row][col];
                tile.setHidden(true);
                tile.setMine(false);
                tile.setNumMineNeighbors(0);
                tile.setText("");
                tile.setBackground(Color.GRAY);
            }
        }
        
        path = RandomPath.getPath(GRID_SIZE);

        score = 0;
        moves = 0;
        scans = STARTING_SCANS;
        lives = STARTING_LIVES;
        
        currentTile = board[0][0];
        currentTile.setText("X");

        sonarScanButton.setEnabled(true);
        
        placePath();
        placeMines();

        // revealAllMines(); // testings

        exposeTile(currentTile);
    }

    private void placeMines() {        
        ArrayList<Point> mineTiles = new ArrayList<>();
        Random r = new Random();
        double percent =
            minePercentageField.getText().matches("^[0-9]*$") ?
            Integer.parseInt(minePercentageField.getText()) 
            : DEFAULT_MINE_PERCENTAGE;

        if (percent >= (100 - path.size())) {
            percent = (100 - path.size()) - 1;
        }

        if (percent <= 0) {
            percent = 0;
        }

        percent = ((double) (percent / 100) * (double) (GRID_SIZE * GRID_SIZE));

        while (mineTiles.size() <= percent) {
            Point temp = new Point(r.nextInt(0, GRID_SIZE), r.nextInt(0, GRID_SIZE));
            if (!(mineTiles.contains(temp)) || mineTiles.isEmpty()) {
                if (!(path.contains(temp))) {
                    mineTiles.add(temp);
                }
            }
        }

        for (var mine : mineTiles) {
            board[mine.getLocation().x][mine.getLocation().y].setMine(true);
        }
    }

    private void placePath() {
        for (var pathTiles : path) {
            board[pathTiles.getLocation().x][pathTiles.getLocation().y].setPath(true);
        }
    }

    private void exposeTile(TileButton tile) {
        int minesAround = 0;

        if (tile.getLocation().x != 0){   
            if (board[tile.getLocation().x-1][tile.getLocation().y].isMine()) { minesAround++; }
        }
        if (tile.getLocation().x != GRID_SIZE - 1) {
            if (board[tile.getLocation().x+1][tile.getLocation().y].isMine()) { minesAround++; }
        }
        if (tile.getLocation().y != 0) {
            if (board[tile.getLocation().x][tile.getLocation().y-1].isMine()) { minesAround++; }
        }
        if (tile.getLocation().y != GRID_SIZE - 1) {
            if (board[tile.getLocation().x][tile.getLocation().y+1].isMine()) { minesAround++; }
        }

        tile.setHidden(false);
        tile.setNumMineNeighbors(minesAround);
        tile.setBackground(getColorForNeighbors(tile.getNumMineNeighbors()));
    }

    private Color getColorForNeighbors(int numNeighbors) {
        return switch (numNeighbors) {
            case 0 -> Color.GREEN;
            case 1 -> Color.YELLOW;
            case 2 -> Color.ORANGE;
            case 3 -> Color.RED;
            default -> Color.BLACK;
        };
    }

    private int getScore() {
        double percent
            = minePercentageField.getText().matches("^[0-9]*$")
            ? Integer.parseInt(minePercentageField.getText()) 
            : DEFAULT_MINE_PERCENTAGE;

        if (percent >= (100 - path.size())) {
            percent = (100 - path.size()) - 1;
        }

        if (percent <= 0) {
            percent = 0;
        }

        int percentInt = (int) percent;

        if (!(moves == path.size() - 1)) {
            if (moves > (path.size() - 1 * 2)) {
                score = 50;
            } else {
                score = 150;
            }
        } else {
            score = 300;
        }
        
        if (lives == STARTING_LIVES) {
            score += 200;
        } else if (lives >= ((STARTING_LIVES / 2) + 1)) {
            score += 100;
        } else if (lives != 0) {
            score += 50;
        } else {
            score -= 50;
        }
        
        if (percentInt == (99 - path.size())) {
            score += 500;
        } else if (percentInt > 50) {
            score += 300;
        } else if (percentInt >= 30) {
            score += 200;
        } else {
            score -= 200;
        }

        score += GRID_SIZE * GRID_SIZE;

        int penalty = 10;
        if (scans != STARTING_SCANS) {
            score /= penalty * (STARTING_SCANS - scans);
        }

        return score;
    }

    private class TileButtonListener implements ActionListener {
        @Override
        public void actionPerformed(ActionEvent e) {
            TileButton clickedTile = (TileButton) e.getSource();

            if (isAdjacent(clickedTile)) {
                if (clickedTile.isMine()) {
                    clickedTile.setHidden(false);
                    clickedTile.setEnabled(false);
                    clickedTile.setBackground(Color.BLACK);
                    lives--;
                    if (lives <= 0) {
                        JOptionPane.showMessageDialog(MineWalkerPanel.this, "Game Over! Final score: " + getScore());
                        revealAllMines();
                        toggleAllTiles(false);
                    }
                } else {
                    exposeTile(clickedTile);
                    currentTile.setText("");
                    currentTile = clickedTile;
                    clickedTile.setText("X");
                    moves++;
                    if (clickedTile.getLocation().x == GRID_SIZE-1 && clickedTile.getLocation().y == GRID_SIZE-1) {
                        JOptionPane.showMessageDialog(MineWalkerPanel.this, "You Win! Final score: " + getScore());
                        revealAllMines();
                        toggleAllTiles(false);
                    }
                }

                livesLabel.setText("Lives: " + lives);
            }
        }

        private boolean isAdjacent(TileButton tile) {
            Point currentPos = currentTile.getLocation();
            Point clickedPos = tile.getLocation();
            return (Math.abs(currentPos.x - clickedPos.x) + Math.abs(currentPos.y - clickedPos.y) == 1);
        }
    }

    private class SonarButtonListener implements ActionListener {
        @Override
        public void actionPerformed(ActionEvent e) {
            int lowerLimitX = -1;
            int lowerLimitY = -1;
            int upperLimitX = 1;
            int upperLimitY = 1;
            if (scans > 0) {
                if (!(currentTile.getLocation().x > 0)) {
                    lowerLimitX = 0;
                }

                if (!(currentTile.getLocation().y > 0)) {
                    lowerLimitY = 0;
                }

                if (!(currentTile.getLocation().x < GRID_SIZE - 1)) {
                    upperLimitX = 0;
                }

                if (!(currentTile.getLocation().y < GRID_SIZE - 1)) {
                    upperLimitY = 0;
                }

                for (int x = lowerLimitX; x <= upperLimitX; x++) {
                    for (int y = lowerLimitY; y <= upperLimitY; y++) {
                        if (!(board[currentTile.getLocation().x + x][currentTile.getLocation().y + y].isMine())) {
                            exposeTile(board[currentTile.getLocation().x + x][currentTile.getLocation().y + y]);
                        } else {
                            board[currentTile.getLocation().x + x][currentTile.getLocation().y + y].setHidden(false);
                            board[currentTile.getLocation().x + x][currentTile.getLocation().y + y].setEnabled(false);
                            board[currentTile.getLocation().x + x][currentTile.getLocation().y + y].setBackground(Color.BLACK);
                        }
                    }   
                }

                scans--;
                sonarScanButton.setText("Remaining scans: " + scans);

                if (scans == 0) {
                    sonarScanButton.setEnabled(false);
                }
            }
        }
    }

    private class NewGameListener implements ActionListener {
        @Override
        public void actionPerformed(ActionEvent e) {
            score = 0;
            moves = 0;
            lives = STARTING_LIVES;
            livesLabel.setText("Lives: " + lives);
            scans = STARTING_SCANS;
            sonarScanButton.setText("Remaining scans: " + scans);
            initializeGame();
        }
    }

    private void toggleAllTiles(boolean bool) {
        for (int row = 0; row < GRID_SIZE; row++) {
            for (int col = 0; col < GRID_SIZE; col++) {
                board[row][col].setEnabled(bool);
            }
        }
    }

    private void revealAllMines() {
        for (int row = 0; row < GRID_SIZE; row++) {
            for (int col = 0; col < GRID_SIZE; col++) {
                if (board[row][col].isMine()) {
                    board[row][col].setBackground(Color.BLACK);
                }
            }
        }
    }
}